<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Products - ReviewPR</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Inter', Arial, sans-serif; background: #f8f9fa; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2rem; border-radius: 12px; margin-bottom: 2rem; }
        .card { background: white; border-radius: 12px; padding: 2rem; margin-bottom: 2rem; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .form-group { margin-bottom: 1.5rem; }
        .form-label { display: block; font-weight: 600; margin-bottom: 0.5rem; color: #374151; }
        .form-control { width: 100%; padding: 0.75rem; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1rem; }
        .form-control:focus { outline: none; border-color: #667eea; }
        .btn { padding: 0.75rem 1.5rem; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #10b981; color: white; }
        .btn-danger { background: #ef4444; color: white; }
        .btn-secondary { background: #6b7280; color: white; }
        .btn:hover { opacity: 0.9; transform: translateY(-1px); }
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem; }
        .product-card { background: white; border-radius: 12px; padding: 1.5rem; box-shadow: 0 2px 4px rgba(0,0,0,0.1); border: 1px solid #e5e7eb; }
        .alert { padding: 1rem; border-radius: 8px; margin-bottom: 1rem; }
        .alert-success { background: #d1fae5; color: #065f46; border: 1px solid #a7f3d0; }
        .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
        .price { font-size: 1.5rem; font-weight: 800; color: #667eea; }
        .stock { padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.875rem; font-weight: 600; }
        .stock.in-stock { background: #d1fae5; color: #065f46; }
        .stock.out-of-stock { background: #fee2e2; color: #991b1b; }
        textarea { resize: vertical; min-height: 100px; }
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.4);
        }
        .modal-content {
            background-color: #fefefe;
            margin: 5% auto;
            padding: 20px;
            border: 1px solid #888;
            width: 80%;
            max-width: 600px;
            border-radius: 12px;
        }
        .close {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        .close:hover {
            color: black;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📦 Product Management System</h1>
            <p>Add, edit, and manage your review services</p>
        </div>

        <div class="nav-links">
            <a href="index.php" class="btn btn-secondary">← Back to Website</a>
            <a href="debug-products.php" class="btn btn-secondary">Debug Products</a>
            <a href="products-simple.php" class="btn btn-secondary">View API</a>
        </div>

        <?php
        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                require_once 'config.php';
                $conn = getSecureConnection();

                if (isset($_POST['action'])) {
                    switch ($_POST['action']) {
                        case 'add':
                            $stmt = $conn->prepare("INSERT INTO products (name, price, category, description, detailed_description, stock, position, image) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                            $stmt->bind_param("sdsssiis", 
                                $_POST['name'], 
                                $_POST['price'], 
                                $_POST['category'], 
                                $_POST['description'], 
                                $_POST['detailed_description'], 
                                $_POST['stock'], 
                                $_POST['position'], 
                                $_POST['image']
                            );
                            
                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Product added successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error adding product: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;

                        case 'update':
                            $stmt = $conn->prepare("UPDATE products SET name=?, price=?, category=?, description=?, detailed_description=?, stock=?, position=?, image=? WHERE id=?");
                            $stmt->bind_param("sdsssiiis", 
                                $_POST['name'], 
                                $_POST['price'], 
                                $_POST['category'], 
                                $_POST['description'], 
                                $_POST['detailed_description'], 
                                $_POST['stock'], 
                                $_POST['position'], 
                                $_POST['image'], 
                                $_POST['id']
                            );
                            
                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Product updated successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error updating product: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;

                        case 'delete':
                            $stmt = $conn->prepare("DELETE FROM products WHERE id=?");
                            $stmt->bind_param("i", $_POST['id']);
                            
                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Product deleted successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error deleting product: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;
                    }
                }

                $conn->close();
            } catch (Exception $e) {
                echo '<div class="alert alert-error">❌ Database Error: ' . $e->getMessage() . '</div>';
            }
        }
        ?>

        <!-- Add New Product Form -->
        <div class="card">
            <h2>➕ Add New Product</h2>
            <form method="POST">
                <input type="hidden" name="action" value="add">
                
                <div class="grid">
                    <div class="form-group">
                        <label class="form-label">Product Name *</label>
                        <input type="text" name="name" class="form-control" required placeholder="e.g., Trustpilot Reviews - Premium">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Price ($) *</label>
                        <input type="number" name="price" step="0.01" class="form-control" required placeholder="9.99">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Category *</label>
                        <select name="category" class="form-control" required>
                            <option value="">Select Category</option>
                            <?php
                            try {
                                require_once 'config.php';
                                $conn = getSecureConnection();
                                $result = $conn->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY position ASC, name ASC");
                                if ($result->num_rows > 0) {
                                    while($category = $result->fetch_assoc()) {
                                        echo '<option value="' . htmlspecialchars($category['name']) . '">' . htmlspecialchars($category['name']) . '</option>';
                                    }
                                }
                                $conn->close();
                            } catch (Exception $e) {
                                // Fallback to hardcoded categories if database fails
                                $categories = ['Trustpilot', 'Google', 'G2', 'Capterra', 'Yelp', 'Facebook', 'Other'];
                                foreach ($categories as $cat) {
                                    echo '<option value="' . $cat . '">' . $cat . '</option>';
                                }
                            }
                            ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Stock Quantity *</label>
                        <input type="number" name="stock" class="form-control" required value="100" min="0">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Position (Order) *</label>
                        <input type="number" name="position" class="form-control" required value="1" min="1">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Image URL</label>
                        <input type="url" name="image" class="form-control" placeholder="https://example.com/image.jpg">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Short Description *</label>
                    <textarea name="description" class="form-control" required placeholder="Brief description of the service..."></textarea>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Detailed Description</label>
                    <textarea name="detailed_description" class="form-control" placeholder="Detailed description with features and benefits..."></textarea>
                </div>
                
                <button type="submit" class="btn btn-success">➕ Add Product</button>
            </form>
        </div>

        <!-- Existing Products -->
        <div class="card">
            <h2>📋 Existing Products</h2>
            
            <?php
            try {
                require_once 'config.php';
                $conn = getSecureConnection();

                $result = $conn->query("SELECT * FROM products ORDER BY position ASC, id ASC");
                
                if ($result->num_rows > 0) {
                    echo '<div class="grid">';
                    
                    while($product = $result->fetch_assoc()) {
                        $stockClass = $product['stock'] > 0 ? 'in-stock' : 'out-of-stock';
                        $stockText = $product['stock'] > 0 ? 'In Stock (' . $product['stock'] . ')' : 'Out of Stock';
                        
                        echo '<div class="product-card">';
                        echo '<h3>' . htmlspecialchars($product['name']) . '</h3>';
                        echo '<div class="price">$' . number_format($product['price'], 2) . '</div>';
                        echo '<p><strong>Category:</strong> ' . htmlspecialchars($product['category']) . '</p>';
                        echo '<p><strong>Position:</strong> ' . $product['position'] . '</p>';
                        echo '<p class="stock ' . $stockClass . '">' . $stockText . '</p>';
                        echo '<p>' . htmlspecialchars(substr($product['description'], 0, 100)) . '...</p>';
                        
                        echo '<div style="margin-top: 1rem;">';
                        echo '<button onclick="editProduct(' . $product['id'] . ')" class="btn btn-primary">✏️ Edit</button> ';
                        echo '<form method="POST" style="display: inline;" onsubmit="return confirm(\'Are you sure you want to delete this product?\')">';
                        echo '<input type="hidden" name="action" value="delete">';
                        echo '<input type="hidden" name="id" value="' . $product['id'] . '">';
                        echo '<button type="submit" class="btn btn-danger">🗑️ Delete</button>';
                        echo '</form>';
                        echo '</div>';
                        
                        echo '</div>';
                    }
                    
                    echo '</div>';
                } else {
                    echo '<p>No products found. Add your first product above!</p>';
                }
                
                $conn->close();
            } catch (Exception $e) {
                echo '<div class="alert alert-error">❌ Error loading products: ' . $e->getMessage() . '</div>';
            }
            ?>
        </div>
    </div>

    <!-- Edit Product Modal -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal()">&times;</span>
            <h2>✏️ Edit Product</h2>
            <form method="POST" id="editForm">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="id" id="edit-id">

                <div class="grid">
                    <div class="form-group">
                        <label class="form-label">Product Name *</label>
                        <input type="text" name="name" id="edit-name" class="form-control" required>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Price ($) *</label>
                        <input type="number" name="price" id="edit-price" step="0.01" class="form-control" required>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Category *</label>
                        <select name="category" id="edit-category" class="form-control" required>
                            <option value="">Select Category</option>
                            <?php
                            try {
                                require_once 'config.php';
                                $conn = getSecureConnection();
                                $result = $conn->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY position ASC, name ASC");
                                if ($result->num_rows > 0) {
                                    while($category = $result->fetch_assoc()) {
                                        echo '<option value="' . htmlspecialchars($category['name']) . '">' . htmlspecialchars($category['name']) . '</option>';
                                    }
                                }
                                $conn->close();
                            } catch (Exception $e) {
                                // Fallback to hardcoded categories if database fails
                                $categories = ['Trustpilot', 'Google', 'G2', 'Capterra', 'Yelp', 'Facebook', 'Other'];
                                foreach ($categories as $cat) {
                                    echo '<option value="' . $cat . '">' . $cat . '</option>';
                                }
                            }
                            ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Stock Quantity *</label>
                        <input type="number" name="stock" id="edit-stock" class="form-control" required min="0">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Position (Order) *</label>
                        <input type="number" name="position" id="edit-position" class="form-control" required min="1">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Image URL</label>
                        <input type="url" name="image" id="edit-image" class="form-control">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Short Description *</label>
                    <textarea name="description" id="edit-description" class="form-control" required></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Detailed Description</label>
                    <textarea name="detailed_description" id="edit-detailed-description" class="form-control"></textarea>
                </div>

                <button type="submit" class="btn btn-primary">💾 Update Product</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            </form>
        </div>
    </div>

    <script>
        function editProduct(id) {
            // Get product data via AJAX
            fetch('get_product.php?id=' + id)
                .then(response => response.json())
                .then(product => {
                    if (product) {
                        // Populate modal with product data
                        document.getElementById('edit-id').value = product.id;
                        document.getElementById('edit-name').value = product.name;
                        document.getElementById('edit-price').value = product.price;
                        document.getElementById('edit-category').value = product.category;
                        document.getElementById('edit-stock').value = product.stock;
                        document.getElementById('edit-position').value = product.position;
                        document.getElementById('edit-image').value = product.image || '';
                        document.getElementById('edit-description').value = product.description;
                        document.getElementById('edit-detailed-description').value = product.detailed_description || '';

                        // Show modal
                        document.getElementById('editModal').style.display = 'block';
                    } else {
                        alert('Product not found!');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading product data!');
                });
        }

        function closeModal() {
            document.getElementById('editModal').style.display = 'none';
        }

        // Close modal when clicking outside of it
        window.onclick = function(event) {
            const modal = document.getElementById('editModal');
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        }

        // Handle form submission for editing
        document.getElementById('editForm').addEventListener('submit', function(e) {
            e.preventDefault(); // Prevent default form submission

            const formData = new FormData(this);

            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('Product updated successfully')) {
                    alert('Product updated successfully!');
                    closeModal();
                    location.reload(); // Refresh page to show updated data
                } else {
                    alert('Error updating product: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error updating product!');
            });
        });
    </script>
</body>
</html>
